/** @file
  The module entry point for Tcg configuration module.

Copyright (c) 2011 - 2017, Intel Corporation. All rights reserved.<BR>
This program and the accompanying materials 
are licensed and made available under the terms and conditions of the BSD License 
which accompanies this distribution.  The full text of the license may be found at 
http://opensource.org/licenses/bsd-license.php

THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS, 
WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.

**/

#include "TcgConfigImpl.h"
#include "Tcg2ConfigNvData.h"
#include <Guid/TpmInstance.h>
#include <Guid/PttInstance.h>
#include <Guid/PttPTPInstanceGuid.h>

#define PTT_BASE_ADDRESS 0xFED40000
extern  EFI_GUID  gSetupVariableGuid;

/**
  The entry point for Tcg configuration driver.

  @param[in]  ImageHandle        The image handle of the driver.
  @param[in]  SystemTable        The system table.

  @retval EFI_ALREADY_STARTED    The driver already exists in system.
  @retval EFI_OUT_OF_RESOURCES   Fail to execute entry point due to lack of resources.
  @retval EFI_SUCCES             All the related protocols are installed on the driver.
  @retval Others                 Fail to install protocols as indicated.

**/
EFI_STATUS
EFIAPI
TcgConfigDriverEntryPoint(
IN EFI_HANDLE          ImageHandle,
IN EFI_SYSTEM_TABLE    *SystemTable
)
{
  EFI_STATUS                Status;
  TCG_CONFIG_PRIVATE_DATA   *PrivateData;
  EFI_TCG_PROTOCOL          *TcgProtocol;
  TCG2_CONFIGURATION        Tcg2Configuration;
  UINTN                     Size;
  UINTN                     Size2;
  TCG_SETUP                 TcgSetup;
  UINT32                    VariableAttributes;
  UINT32                    VariableAttributes2;

//
// RPPO-KBL-0083: RoyalParkOverrideBegin
//
  Size2= sizeof (Tcg2Configuration);
  Status = gRT->GetVariable (
                  TCG2_STORAGE_NAME,
                  &gTcg2ConfigFormSetGuid,
                  &VariableAttributes2,
                  &Size2,
                  &Tcg2Configuration
                  );

  if (EFI_ERROR (Status)) {
    VariableAttributes2 = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS;
  }
//
// RPPO-KBL-0083: RoyalParkOverrideEnd
//
  Size = sizeof (TCG_SETUP);
  Status = gRT->GetVariable(
                  TCG_SETUP_NAME,
                  &gSetupVariableGuid,
                  &VariableAttributes,
                  &Size,
                  &TcgSetup
                  );
  ASSERT_EFI_ERROR(Status);

  DEBUG((DEBUG_INFO, "Get Tpm Setup option variable = %r \n", Status));

  TcgSetup.dTpm12Present = 0;
  TcgSetup.dTpm20Present = 0;
  TcgSetup.PttPresent = 0;

  if ((BOOLEAN)(*(UINT8 *)(UINTN)TPM_BASE_ADDRESS != 0xff)) {

    if (CompareGuid(PcdGetPtr(PcdTpmInstanceGuid), &gEfiTpmDeviceInstanceTpm12Guid)) {
      TcgSetup.dTpm12Present = 1;
    }

    if (CompareGuid(PcdGetPtr(PcdTpmInstanceGuid), &gEfiTpmDeviceInstanceTpm20DtpmGuid)) {
      TcgSetup.dTpm20Present = 1;
    }
  }

  if ((*(UINT8 *)(UINTN)(TPM_BASE_ADDRESS + 0x30) != 0xFF)) {
    if (CompareGuid(PcdGetPtr(PcdTpmInstanceGuid), &gTpmDeviceInstanceTpm20PttPtpGuid)) {
      TcgSetup.PttPresent = 1;
    }
  }

  Status = gRT->SetVariable(
                  TCG_SETUP_NAME,
                  &gSetupVariableGuid,
                  VariableAttributes,
                  Size,
                  &TcgSetup
                  );
  DEBUG((DEBUG_INFO, "Set TcgSetup variable = %r \n", Status));

  if (CompareGuid(PcdGetPtr(PcdTpmInstanceGuid), &gEfiTpmDeviceInstanceNoneGuid)){
    Size2= sizeof(Tcg2Configuration);
    Status = gRT->GetVariable (
                    TCG2_STORAGE_NAME,
                    &gTcg2ConfigFormSetGuid,
                    &VariableAttributes2,
                    &Size2,
                    &Tcg2Configuration
                    );

    //
    // No TPM available
    //
    Tcg2Configuration.TpmDevice           = TPM_DEVICE_NULL;

    //
    // Save to variable so platform driver can get it.
    //
    Status = gRT->SetVariable (
                    TCG2_STORAGE_NAME,
                    &gTcg2ConfigFormSetGuid,
                    VariableAttributes2,
                    Size2,
                    &Tcg2Configuration
                    );
    if (EFI_ERROR (Status)) {
      DEBUG ((EFI_D_ERROR, "TcgConfigDriver: Fail to set TCG2_STORAGE_NAME\n"));
    }

    return EFI_UNSUPPORTED;
  }

  if (!CompareGuid(PcdGetPtr(PcdTpmInstanceGuid), &gEfiTpmDeviceInstanceTpm12Guid)){
    DEBUG((DEBUG_ERROR, "No TPM12 instance required!\n"));
    return EFI_UNSUPPORTED;
  } else {
    Tcg2Configuration.TpmDevice = TPM_DEVICE_1_2;

    //
    // Save to variable so platform driver can get it.
    //
    Status = gRT->SetVariable(
                    TCG2_STORAGE_NAME,
                    &gTcg2ConfigFormSetGuid,
                    VariableAttributes2, // RPPO-KBL-0083
                    sizeof(Tcg2Configuration),
                    &Tcg2Configuration
                    );
    if (EFI_ERROR(Status)) {
      DEBUG((EFI_D_ERROR, "TcgConfigDriver: Fail to set TCG2_STORAGE_NAME\n"));
    } else {
      DEBUG((EFI_D_INFO, "TcgConfigDriver: Successfully set TCG2_STORAGE_NAME\n"));
    }
  }
  Status = TisPcRequestUseTpm ((TIS_TPM_HANDLE) (UINTN) TPM_BASE_ADDRESS);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TPM not detected!\n"));
    return Status;
  }

  Status = gBS->LocateProtocol (&gEfiTcgProtocolGuid, NULL, (VOID **) &TcgProtocol);
  if (EFI_ERROR (Status)) {
    TcgProtocol = NULL;
  }
  
  Status = gBS->OpenProtocol (
                  ImageHandle,
                  &gEfiCallerIdGuid,
                  NULL,
                  ImageHandle,
                  ImageHandle,
                  EFI_OPEN_PROTOCOL_TEST_PROTOCOL
                  );
  if (!EFI_ERROR (Status)) {
    return EFI_ALREADY_STARTED;
  }
  
  //
  // Create a private data structure.
  //
  PrivateData = AllocateCopyPool (sizeof (TCG_CONFIG_PRIVATE_DATA), &mTcgConfigPrivateDateTemplate);
  if (PrivateData == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  
  PrivateData->TcgProtocol = TcgProtocol;
  
  //
  // Install TCG configuration form
  //
  Status = InstallTcgConfigForm (PrivateData);
  if (EFI_ERROR (Status)) {
    goto ErrorExit;
  }

  //
  // Install private GUID.
  //    
  Status = gBS->InstallMultipleProtocolInterfaces (
                  &ImageHandle,
                  &gEfiCallerIdGuid,
                  PrivateData,
                  NULL
                  );

  if (EFI_ERROR (Status)) {
    goto ErrorExit;
  }

  return EFI_SUCCESS;

ErrorExit:
  if (PrivateData != NULL) {
    UninstallTcgConfigForm (PrivateData);
  }  
  
  return Status;
}

/**
  Unload the Tcg configuration form.

  @param[in]  ImageHandle         The driver's image handle.

  @retval     EFI_SUCCESS         The Tcg configuration form is unloaded.
  @retval     Others              Failed to unload the form.

**/
EFI_STATUS
EFIAPI
TcgConfigDriverUnload (
  IN EFI_HANDLE  ImageHandle
  )
{
  EFI_STATUS                  Status;
  TCG_CONFIG_PRIVATE_DATA   *PrivateData;

  Status = gBS->HandleProtocol (
                  ImageHandle,
                  &gEfiCallerIdGuid,
                  (VOID **) &PrivateData
                  );  
  if (EFI_ERROR (Status)) {
    return Status;  
  }
  
  ASSERT (PrivateData->Signature == TCG_CONFIG_PRIVATE_DATA_SIGNATURE);

  gBS->UninstallMultipleProtocolInterfaces (
         &ImageHandle,
         &gEfiCallerIdGuid,
         PrivateData,
         NULL
         );
  
  UninstallTcgConfigForm (PrivateData);

  return EFI_SUCCESS;
}
